/*
    Copyright (C) 1996-2008 by Jan Eric Kyprianidis <www.kyprianidis.com>
    All rights reserved.
    
    This program is free  software: you can redistribute it and/or modify 
    it under the terms of the GNU Lesser General Public License as published 
    by the Free Software Foundation, either version 2.1 of the License, or 
    (at your option) any later version.

    Thisprogram  is  distributed in the hope that it will be useful, 
    but WITHOUT ANY WARRANTY; without even the implied warranty of 
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
    GNU Lesser General Public License for more details.
    
    You should  have received a copy of the GNU Lesser General Public License
    along with  this program; If not, see <http://www.gnu.org/licenses/>. 
*/
#include <lib3ds.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <antiprism/antiprism.h>

/** 
	@example cube.c
	This examples demonstrates how to export a textured cube using lib3ds.
*/


/*
static float g_vertices[8][3] = {
    { -10.0, -10.0,  15.0 },
    {  10.0, -10.0,  15.0 },
    {  10.0,  10.0,  15.0 },
    { -10.0,  10.0,  15.0 },
    { -10.0, -10.0, -15.0 },
    {  10.0, -10.0, -15.0 },
    {  10.0,  10.0, -15.0 },
    { -10.0,  10.0, -15.0 }
};


// CCW
static unsigned short g_indices[12][3] = {
    { 0, 5, 1 },
    { 0, 4, 5 },
    { 1, 6, 2 },
    { 1, 5, 6 },
    { 2, 6, 7 },
    { 2, 7, 3 },
    { 0, 3, 7 },
    { 0, 7, 4 },
    { 0, 1, 2 },
    { 0, 2, 3 },
    { 4, 7, 6 },
    { 4, 6, 5 }
};
*/

class off3ds_opts: public prog_opts {
   public:
      vector<string> ifiles;
      string ofile;

      off3ds_opts(): prog_opts("off_to_3ds")
                 {}
      void process_command_line(int argc, char **argv);
      void usage();
};


void off3ds_opts::usage()
{
   fprintf(stdout,
"\n"
"Usage: %s [options] input_files\n"
"\n"
"Read one or more files in OFF format, combine them into a single file and\n"
"output in 3ds format. input_files is the list of files to process, or if\n"
"not given read from standard input\n"
"\n"
"Options\n"
"%s"
"  -o <file> write output to file (default: write to standard output)\n"
"\n"
"\n", prog_name(), help_ver_text);
}

   
void off3ds_opts::process_command_line(int argc, char **argv)
{
   //char errmsg[MSG_SZ];
   opterr = 0;
   char c;

   handle_long_opts(argc, argv);

   while( (c = getopt(argc, argv, ":ho:")) != -1 ) {
      if(common_opts(c, optopt))
         continue;

      switch(c) {
         case 'o':
            ofile = optarg;
            break;

         default:
            error("unknown command line error");
      }
   }

   while(argc-optind)
      ifiles.push_back(string(argv[optind++]));
}



Lib3dsFile *get_file_3ds(const geom_if &geom, char *errmsg=0)
{
   if(errmsg)
      *errmsg = '\0';

   Lib3dsFile *file = lib3ds_file_new();
   file->frames = 360;

   Lib3dsMesh *mesh = lib3ds_mesh_new("off_to_3ds");
   Lib3dsMeshInstanceNode *inst;        
   lib3ds_file_insert_mesh(file, mesh, -1);

   lib3ds_mesh_resize_vertices(mesh, geom.verts().size(), 1, 0);
   for(int i=0; i<(int)geom.verts().size(); ++i) {
      float pt[3];
      for(int j=0; j<3; j++)
         pt[j] = (float)geom.verts(i)[j];
      lib3ds_vector_copy(mesh->vertices[i], pt);
   }

   lib3ds_mesh_resize_faces(mesh, geom.faces().size());
   for(int i=0; i<(int)geom.faces().size(); ++i) {
      for(int j=0; j<3; ++j) {
         mesh->faces[i].index[j] = geom.faces(i, j);
      }
   }

   inst = lib3ds_node_new_mesh_instance(mesh, "01", NULL, NULL, NULL);
   lib3ds_file_append_node(file, (Lib3dsNode*)inst, NULL);
   return file;
}




int main(int argc, char **argv)
{
   off3ds_opts opts;
   opts.process_command_line(argc, argv);
   if(!opts.ifiles.size())
      opts.ifiles.push_back("");

   char errmsg[MSG_SZ]="";
   col_geom_v geoms;
   for(unsigned int i=0; i<opts.ifiles.size(); i++) {
      col_geom_v geom;
      if(!geom.read(opts.ifiles[i], errmsg))
         opts.error(errmsg);
      if(*errmsg)
         opts.warning(errmsg);
      geoms.append(geom);
   }

   geoms.triangulate();

   Lib3dsFile *file = get_file_3ds(geoms);
   
   const char *tmp_fn = "antiprism_off_to_3ds_tmp.off";
   bool use_stdout = (opts.ofile=="" || opts.ofile=="-");
   const char *out_file = (use_stdout) ? tmp_fn : opts.ofile.c_str();
   
   if (!lib3ds_file_save(file, out_file)) {
      fprintf(stderr, "ERROR: Saving 3ds file failed!\n");
   }
   
   if(use_stdout) {
      FILE *fp = fopen(tmp_fn, "r");
      if(fp == NULL)
         opts.error("cannot open temporary file to write to standard output");
      
      int c;
      while((c = fgetc(fp)) !=EOF)
         putc(c, stdout);
   }

   lib3ds_file_free(file);
}
